/**
 * PAINEL DE RECARGAS 1.0 - Remix Automation Service
 * Automação de Remix Lovable com seletores atualizados
 * Fluxo: Login → Acessa projeto → Remix → Publish → Alterar nome → Repetir
 */

const { chromium } = require('playwright');
const notificationService = require('./notificationService');

// State management
let broadcastFn = null;
let running = false;
let config = {};
let browser = null;
let currentIteration = 0;
let totalCredits = 0;
const CREDITS_PER_CYCLE = 10;

// User agents reais para rotação
const USER_AGENTS = [
    'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
    'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/119.0.0.0 Safari/537.36',
    'Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:121.0) Gecko/20100101 Firefox/121.0',
    'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36 Edg/120.0.0.0',
];

function setBroadcast(fn) {
    broadcastFn = fn;
}

function updateConfig(newConfig) {
    config = { ...config, ...newConfig };
}

function log(message, level = 'info') {
    try {
        const timestamp = new Date().toLocaleTimeString('pt-BR');
        const logEntry = {
            timestamp,
            message: String(message),
            level: String(level || 'info')
        };

        console.log(`[Recargas 1.0] ${message}`);

        if (typeof broadcastFn === 'function') {
            broadcastFn('remix-log', logEntry);
        } else {
            console.error('[Error] broadcastFn is not a function/initialized');
        }
    } catch (e) {
        console.error('[Log Error]', e);
    }
}

function updateStatus(status, step = null, iteration = null) {
    if (broadcastFn) {
        broadcastFn('remix-status', {
            status,
            step,
            iteration,
            totalIterations: config.remixRepeatCount || 1
        });
    }
}

function isRunning() {
    return running;
}

function stop() {
    running = false;
    currentIteration = 0;
    log('🛑 Automação parada pelo usuário', 'warn');
    updateStatus('stopped');

    if (browser) {
        browser.close().catch(() => { });
        browser = null;
    }
}

async function stealthPage(page) {
    await page.addInitScript(() => {
        Object.defineProperty(navigator, 'webdriver', {
            get: () => undefined
        });

        Object.defineProperty(navigator, 'plugins', {
            get: () => [
                { name: 'Chrome PDF Plugin', filename: 'internal-pdf-viewer' },
                { name: 'Chrome PDF Viewer', filename: 'mhjfbmdgcfjbbpaeojofohoefgiehjai' },
                { name: 'Native Client', filename: 'internal-nacl-plugin' }
            ]
        });

        Object.defineProperty(navigator, 'languages', {
            get: () => ['pt-BR', 'pt', 'en-US', 'en']
        });

        window.chrome = {
            runtime: {},
            loadTimes: function () { },
            csi: function () { },
            app: {}
        };

        const originalQuery = window.navigator.permissions.query;
        window.navigator.permissions.query = (parameters) => (
            parameters.name === 'notifications' ?
                Promise.resolve({ state: Notification.permission }) :
                originalQuery(parameters)
        );

        delete window.cdc_adoQpoasnfa76pfcZLmcfl_Array;
        delete window.cdc_adoQpoasnfa76pfcZLmcfl_Promise;
        delete window.cdc_adoQpoasnfa76pfcZLmcfl_Symbol;
    });
}

async function start(remixConfig) {
    if (running) {
        log('⚠️ Automação já está em execução', 'warn');
        return;
    }

    running = true;
    currentIteration = 0;
    config = { ...config, ...remixConfig };

    // Inicializar serviço de notificações (TELEGRAM)
    notificationService.init({
        botToken: config.telegramBotToken,
        chatId: config.telegramChatId
    });

    const {
        projectUrl,
        remixRepeatCount,
        baseName,
        headless,
        email,
        password,
        instanceUrl
    } = config;

    // Validação de campos obrigatórios
    if (!email || !password) {
        log('❌ Email e senha são obrigatórios!', 'error');
        updateStatus('error', 'Email e senha são obrigatórios');
        running = false;
        return;
    }

    if (!projectUrl) {
        log('❌ URL do projeto é obrigatória!', 'error');
        updateStatus('error', 'URL do projeto é obrigatória');
        running = false;
        return;
    }

    log(`\n🚀 ========== PAINEL DE RECARGAS 1.0 - INICIANDO ==========`);
    log(`📧 Email: ${email}`);
    log(`📎 URL do Projeto: ${projectUrl}`);
    log(`🔁 Repetições: ${remixRepeatCount}`);
    log(`📝 Nome base: ${baseName}`);
    log(`🖥️ Instância: ${instanceUrl}`);

    // Resetar contador de créditos
    totalCredits = 0;
    log(`💰 Contador de créditos resetado`);

    try {
        updateStatus('starting', 'Iniciando navegador...');

        const userAgent = USER_AGENTS[Math.floor(Math.random() * USER_AGENTS.length)];

        browser = await chromium.launch({
            channel: 'msedge',
            headless: headless || false,
            args: [
                '--start-maximized',
                '--disable-blink-features=AutomationControlled',
                '--disable-infobars',
                '--no-default-browser-check',
                '--no-first-run',
                '--disable-dev-shm-usage',
                '--no-sandbox',
                '--disable-setuid-sandbox',
                '--disable-background-timer-throttling',
                '--disable-backgrounding-occluded-windows',
                '--disable-renderer-backgrounding'
            ]
        });

        const context = await browser.newContext({
            viewport: null, // null para usar o tamanho real da janela maximizada
            userAgent,
            locale: 'pt-BR',
            timezoneId: 'America/Sao_Paulo',
        });

        const page = await context.newPage();
        await stealthPage(page);
        page.setDefaultTimeout(60000);

        // ==================== LOGIN ====================
        updateStatus('running', 'Verificando login...', 0);
        log('🔑 Acessando página de login...');

        await page.goto('https://lovable.dev/login');

        // Aguardar um pouco para verificar redirecionamentos
        try {
            await page.waitForLoadState('domcontentloaded');
            await page.waitForTimeout(3000);
        } catch (e) { }

        const currentUrl = page.url();
        log(`📍 URL atual: ${currentUrl}`);

        // Verificação Inteligente: Já estamos logados?
        // Se a URL não for /login OU se encontrarmos um avatar/indicador de logado
        let alreadyLoggedIn = false;

        if (!currentUrl.includes('/login')) {
            alreadyLoggedIn = true;
            log('✅ Redirecionado automaticamente (Provavelmente já logado)');
        } else {
            // Tentar identificar se estamos na tela de login ou dashboard
            try {
                // Se encontrar o input de email, NÃO está logado.
                // Se encontrar avatar, ESTÁ logado.
                const emailCheck = page.getByRole('textbox', { name: 'E-mail' });
                if (await emailCheck.isVisible()) {
                    alreadyLoggedIn = false;
                } else {
                    // Talvez tenha redirecionado mas a URL ainda não mudou ou é single page app
                    // Vamos tentar achar elemento de dashboard
                    const dashboardEl = page.locator('header').first();
                    if (await dashboardEl.isVisible()) {
                        alreadyLoggedIn = true;
                    }
                }
            } catch (e) {
                // Em dúvida, assume que precisa logar ou falhará no passo seguinte
            }
        }

        if (alreadyLoggedIn) {
            log('✅ Sessão detectada! Pulando etapa de login...');
        } else {
            // LOGIN NECESSÁRIO

            // Preencher e-mail
            log('📧 Preenchendo e-mail...');
            const emailInput = page.getByRole('textbox', { name: 'E-mail' });
            await emailInput.waitFor({ state: 'visible', timeout: 15000 });
            await emailInput.click();
            await emailInput.fill(email);

            // Clicar em Continuar
            log('➡️ Clicando em Continuar...');
            const continueBtn = page.getByRole('button', { name: 'Continuar', exact: true });
            await continueBtn.waitFor({ state: 'visible', timeout: 10000 });
            await continueBtn.click();

            // Preencher senha
            log('🔒 Preenchendo senha...');
            const passwordInput = page.getByRole('textbox', { name: 'Senha' });
            await passwordInput.waitFor({ state: 'visible', timeout: 15000 });
            await passwordInput.click();
            await passwordInput.fill(password);

            // Clicar em Login
            log('🚪 Clicando em Login...');
            const loginBtn = page.getByRole('button', { name: 'Login' });
            await loginBtn.waitFor({ state: 'visible', timeout: 10000 });
            await loginBtn.click();
        }

        // Aguardar 10 segundos e acessar URL do projeto
        log('⏳ Aguardando 10 segundos...');
        await page.waitForTimeout(10000);

        log('📍 Acessando URL do projeto...');
        await page.goto(projectUrl);
        await page.waitForLoadState('domcontentloaded');

        log('✅ Login realizado com sucesso!');

        if (broadcastFn) {
            broadcastFn('remix-login-success', { email });
        }

        // ==================== LOOP PRINCIPAL DE REMIX ====================
        for (let i = 1; i <= remixRepeatCount && running; i++) {
            currentIteration = i;
            const iterationName = `${baseName}${i + 1}`;

            log(`\n🔄 ========== ITERAÇÃO ${i}/${remixRepeatCount} ==========`);
            updateStatus('running', `Iteração ${i}/${remixRepeatCount}`, i);

            // ==================== NAVEGAR PARA O PROJETO ====================
            updateStatus('running', `[${i}/${remixRepeatCount}] Acessando projeto...`, i);
            log('📍 Navegando para o projeto...');

            await page.goto(projectUrl);
            await page.waitForLoadState('domcontentloaded');

            // Aguardar página carregar completamente
            log('⏳ Aguardando página carregar...');
            await page.waitForTimeout(5000);

            // Aguardar até que algum botão "Remix" apareça na página
            try {
                await page.waitForSelector('button:has-text("Remix")', { timeout: 15000 });
                log('✅ Botão Remix detectado na página!');
            } catch (e) {
                log('⚠️ Botão não detectado, recarregando página...', 'warn');
                await page.reload();
                await page.waitForLoadState('domcontentloaded');
                await page.waitForTimeout(5000);
            }

            // ==================== REMIX ====================
            updateStatus('running', `[${i}/${remixRepeatCount}] Fazendo Remix...`, i);
            log('🔀 Procurando botão Remix PRINCIPAL (no header)...');

            // IMPORTANTE: Fechar qualquer menu aberto antes de procurar o botão
            try {
                await page.keyboard.press('Escape');
                await page.waitForTimeout(500);
            } catch (e) { }

            // Tentar encontrar o botão Remix com múltiplas estratégias
            // PRIORIDADE: Botão no HEADER (canto superior direito), NÃO os itens de menu
            let remixBtn = null;
            let remixFound = false;

            // Estratégia 1: Botão Remix que NÃO está em um menu (mais preciso)
            // O botão principal fica no header, não em dropdowns ou menus
            try {
                // Seletor específico: botão com texto "Remix" que NÃO é um menuitem
                remixBtn = page.locator('button:has-text("Remix"):not([role="menuitem"])').first();
                await remixBtn.waitFor({ state: 'visible', timeout: 5000 });

                // Verificar se é realmente um botão primário (não um item de menu)
                const buttonClasses = await remixBtn.getAttribute('class');
                if (buttonClasses && (buttonClasses.includes('bg-primary') || buttonClasses.includes('bg-black'))) {
                    remixFound = true;
                    log('✅ Botão Remix PRINCIPAL encontrado (header, cor primária)');
                } else {
                    // Continuar procurando
                    remixFound = false;
                }
            } catch (e) {
                log('⚠️ Seletor primário falhou, tentando alternativo...', 'warn');
            }

            // Estratégia 2: Botão Remix com classes específicas de botão primário
            if (!remixFound) {
                try {
                    remixBtn = page.locator('button.bg-primary:has-text("Remix"), button.bg-black:has-text("Remix")').first();
                    await remixBtn.waitFor({ state: 'visible', timeout: 5000 });
                    remixFound = true;
                    log('✅ Botão Remix encontrado (classe bg-primary/bg-black)');
                } catch (e) {
                    log('⚠️ Seletor de classe falhou, tentando role...', 'warn');
                }
            }

            // Estratégia 3: Por role button com nome EXATO "Remix" (não "Remix this project")
            if (!remixFound) {
                try {
                    remixBtn = page.getByRole('button', { name: 'Remix', exact: true }).first();
                    await remixBtn.waitFor({ state: 'visible', timeout: 5000 });
                    remixFound = true;
                    log('✅ Botão Remix encontrado (role exact)');
                } catch (e) {
                    log('⚠️ Seletor role exato falhou, tentando próximo...', 'warn');
                }
            }

            // Estratégia 4: Fallback genérico
            if (!remixFound) {
                try {
                    remixBtn = page.locator('[data-testid*="remix"], .remix-button').first();
                    await remixBtn.waitFor({ state: 'visible', timeout: 8000 });
                    remixFound = true;
                    log('✅ Botão Remix encontrado (fallback)');
                } catch (e3) {
                    log('❌ Nenhum botão Remix encontrado!', 'error');
                }
            }

            // Se não encontrou, pular iteração
            if (!remixFound || !remixBtn) {
                log('❌ Botão Remix não localizado, pulando iteração...', 'error');
                continue;
            }

            // Verificação final: garantir que não é um item de menu
            try {
                const role = await remixBtn.getAttribute('role');
                if (role === 'menuitem') {
                    log('⚠️ Detectado menuitem em vez de button! Corrigindo...', 'warn');
                    // Fechar menu e tentar de novo
                    await page.keyboard.press('Escape');
                    await page.waitForTimeout(500);
                    remixBtn = page.getByRole('button', { name: 'Remix', exact: true }).first();
                    await remixBtn.waitFor({ state: 'visible', timeout: 5000 });
                }
            } catch (e) { }

            // Clicar no botão Remix
            log('🔀 Clicando em Remix...');
            try {
                await remixBtn.click({ timeout: 5000 });
            } catch (clickError) {
                log(`⚠️ Clique normal falhou, tentando forçado...`, 'warn');
                await remixBtn.click({ timeout: 10000, force: true });
            }

            // Aguardar modal aparecer e clicar no segundo botão Remix (confirmação)
            log('🔀 Aguardando modal de confirmação...');

            // Procurar botão de confirmação dentro do modal
            try {
                try {
                    await page.waitForSelector('[role="dialog"], .modal, [data-state="open"]', { timeout: 10000 });
                } catch (modalError) {
                    log('ℹ️ Modal não detectado explícitamente, tentando achar botões direto...', 'warn');
                }

                let confirmBtn = null;

                // 1. Tentar botão com mesmo texto "Remix"
                try {
                    confirmBtn = page.getByRole('button', { name: 'Remix' }).last();
                    await confirmBtn.waitFor({ state: 'visible', timeout: 5000 });
                    log('🔀 Botão de confirmação "Remix" encontrado');
                } catch (e) {
                    // 2. Tentar "Create project" ou "Create"
                    try {
                        confirmBtn = page.getByRole('button', { name: /Create|Criar|Confirm/i }).first();
                        await confirmBtn.waitFor({ state: 'visible', timeout: 5000 });
                        log('🔀 Botão de confirmação alternativo encontrado');
                    } catch (e2) {
                        try {
                            confirmBtn = page.locator('[role="dialog"] button[type="submit"], [role="dialog"] button.bg-primary').last();
                            await confirmBtn.waitFor({ state: 'visible', timeout: 5000 });
                            log('🔀 Botão de confirmação genérico encontrado');
                        } catch (e3) {
                            throw new Error('Botão de confirmação não encontrado');
                        }
                    }
                }

                if (confirmBtn) {
                    log('🔀 Clicando na confirmação...');
                    await confirmBtn.click({ force: true });
                }

            } catch (e) {
                log(`⚠️ Erro no processo de confirmação: ${e.message}`, 'warn');
            }

            // Aguardar redirecionamento para o novo projeto
            log('⏳ Aguardando novo projeto...');
            try {
                await page.waitForURL(/\/projects\/[a-f0-9-]+/, { timeout: 60000 });
                log('✅ Redirecionado para novo projeto!');
            } catch (e) {
                log('⚠️ Timeout aguardando projeto, verificando URL atual...', 'warn');
                const currentUrl = page.url();
                log(`📍 URL atual: ${currentUrl}`);
                if (!currentUrl.includes('/projects/')) {
                    log('❌ Remix pode ter falhado, pulando iteração...', 'error');
                    continue;
                }
            }

            await page.waitForLoadState('domcontentloaded');
            log('✅ Remix concluído!');

            if (broadcastFn) {
                broadcastFn('remix-remix-success', { iteration: i });
            }

            // ==================== PUBLISH ====================
            updateStatus('running', `[${i}/${remixRepeatCount}] Publicando...`, i);
            log('📤 Clicando em Publish...');

            try {
                const publishBtn = page.getByRole('button', { name: 'Publish' });
                await publishBtn.waitFor({ state: 'visible', timeout: 30000 });
                await publishBtn.click();

                // Aguardar segundo botão Publish aparecer e clicar
                log('📤 Confirmando Publish...');
                const confirmPublishBtn = page.getByRole('button', { name: 'Publish' }).last();
                await confirmPublishBtn.waitFor({ state: 'visible', timeout: 15000 });
                await confirmPublishBtn.click();

                // Aguardar e fechar modal de sucesso
                log('⏳ Aguardando publicação...');
                try {
                    const closeBtn = page.getByRole('button', { name: 'Close' });
                    await closeBtn.waitFor({ state: 'visible', timeout: 30000 });
                    await closeBtn.click();
                    log('✅ Publicação concluída!');
                } catch (e) {
                    log('ℹ️ Modal Close não encontrado, continuando...', 'warn');
                }
            } catch (publishError) {
                log(`⚠️ Erro no Publish: ${publishError.message}`, 'warn');
            }

            if (broadcastFn) {
                broadcastFn('remix-publish-success', { iteration: i });
            }

            // ==================== ALTERAR NOME (usando seletores exatos) ====================
            log('✏️ Alterando nome...');

            // Garantir que estamos na página de configurações correta
            const settingsUrl = 'https://lovable.dev/settings?tab=account';
            if (page.url() !== settingsUrl) {
                log(`📍 Navegando para Settings: ${settingsUrl}`);
                await page.goto(settingsUrl);
                await page.waitForLoadState('domcontentloaded');
            }

            try {
                await page.waitForTimeout(2000);

                // PASSO 1: Encontrar o campo Name usando o seletor EXATO
                log('🔍 Procurando campo Name (input[name="display_name"])...');
                const nameInput = page.locator('input[name="display_name"]');
                await nameInput.waitFor({ state: 'visible', timeout: 5000 });

                // Limpar e preencher o campo
                await nameInput.click();
                await nameInput.fill('');
                await nameInput.fill(iterationName);
                log(`✅ Nome preenchido: ${iterationName}`);

                await page.waitForTimeout(500);

                // PASSO 2: Clicar no botão "Atualizar" usando o seletor EXATO
                log('💾 Clicando no botão Atualizar...');
                const atualizarBtn = page.locator('button[type="submit"][form="user-form-section"]');

                if (await atualizarBtn.isVisible({ timeout: 3000 })) {
                    await atualizarBtn.click();
                    log('✅ Botão Atualizar clicado!');
                    await page.waitForTimeout(2000);
                    log(`✅ Nome alterado para: ${iterationName}`);
                } else {
                    // Fallback: tentar qualquer botão com texto "Atualizar"
                    const fallbackBtn = page.locator('button:has-text("Atualizar")').first();
                    if (await fallbackBtn.isVisible({ timeout: 2000 })) {
                        await fallbackBtn.click();
                        log('✅ Botão Atualizar (fallback) clicado!');
                    } else {
                        await nameInput.press('Enter');
                    }
                }

                if (broadcastFn) {
                    broadcastFn('remix-name-changed', { iteration: i, name: iterationName });
                }

            } catch (e) {
                log(`⚠️ Erro ao alterar nome: ${e.message}`, 'warn');
                const url = page.url();
                log(`URL atual no erro: ${url}`, 'warn');
            }

            // ==================== FINALIZAÇÃO DA ITERAÇÃO ====================
            totalCredits += CREDITS_PER_CYCLE;
            log(`💰 Créditos enviados neste ciclo: ${CREDITS_PER_CYCLE} | Total: ${totalCredits}`);

            if (broadcastFn) {
                broadcastFn('remix-credits-update', { credits: totalCredits });
            }

            log(`✅ ========== ITERAÇÃO ${i} CONCLUÍDA ==========\n`);

            // [NEW] Notificação a cada ciclo (TELEGRAM)
            await notificationService.sendTelegram(
                `🔄 <b>Ciclo Concluído!</b>\n\n` +
                `🖥️ <b>Painel:</b> ${instanceUrl || 'Local'}\n` +
                `🔢 <b>Progresso:</b> ${i}/${remixRepeatCount}\n` +
                `💰 <b>Créditos Gerados:</b> ${totalCredits}\n` +
                `📝 <b>Nome:</b> ${iterationName}`
            );

            if (broadcastFn) {
                broadcastFn('remix-iteration-complete', {
                    iteration: i,
                    total: remixRepeatCount,
                    name: iterationName
                });
            }
        }

        // ==================== FINALIZAÇÃO ====================
        if (running) {
            log(`\n🎉 ========== AUTOMAÇÃO CONCLUÍDA ==========`);
            log(`✅ Total de ${remixRepeatCount} remix(es) realizado(s)!`);
            log(`💰 Total de créditos enviados: ${totalCredits}`);
            updateStatus('completed', 'Automação concluída!', remixRepeatCount);

            // Notificação de Sucesso Final (TELEGRAM)
            await notificationService.sendTelegram(
                `✅ <b>Painel Finalizado!</b>\n\n` +
                `🖥️ <b>Painel:</b> ${instanceUrl || 'Local'}\n` +
                `🚀 <b>Ciclos Totais:</b> ${remixRepeatCount}\n` +
                `💰 <b>Total Créditos:</b> ${totalCredits}\n` +
                `📝 <b>Base:</b> ${baseName}`
            );
        }

    } catch (error) {
        log(`❌ Erro na automação: ${error.message}`, 'error');
        console.error('Stack:', error.stack);
        updateStatus('error', error.message);

        // Notificação de Erro (TELEGRAM)
        await notificationService.sendTelegram(
            `❌ <b>Erro no Painel de Recargas!</b>\n\n` +
            `🖥️ <b>Painel:</b> ${config.instanceUrl || 'Local'}\n` +
            `⚠️ <b>Erro:</b> ${error.message}\n` +
            `📍 <b>Iteração:</b> ${currentIteration}\n` +
            `💰 <b>Créditos Salvos:</b> ${totalCredits}`
        );

    } finally {
        running = false;

        // Função auxiliar de Login


        if (browser) {
            log('⏳ Fechando navegador...');
            await browser.close().catch(() => { });
            browser = null;
        }
    }
}

module.exports = {
    setBroadcast,
    updateConfig,
    start,
    stop,
    isRunning
};
